<?php

declare(strict_types=1);

namespace Stream\Infrastructure\Http\Controller\Api;

use Core\Framework\Http\Controller\Api\BaseController;
use Core\General\Translation\Service\Translator;
use Stream\Infrastructure\Mapper\TimeDetailsMapper;
use Stream\Infrastructure\Service\AudioSegmentManager;
use Stream\Infrastructure\Service\YammatStreamManager;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\Routing\Annotation\Route;

#[
    Route(
        path: '/stream',
        name: 'stream.',
    ),
]
class StreamController extends BaseController
{
    public function __construct(
        Translator $translator,
        private readonly YammatStreamManager $yammatStreamManager,
        private readonly AudioSegmentManager $audioSegmentManager,
        private readonly TimeDetailsMapper $timeDetailsMapper,
    ) {
        parent::__construct($translator);
    }

    #[
        Route(
            path: '/{date}/{time}',
            name: 'stream-chunk',
            methods: ['GET'],
            priority: 10,
        ),
    ]
    public function streamChunk(string $date, string $time): Response
    {
        $dtString = \sprintf('%s %s', $date, $time);
        try {
            $timeDetails = $this->timeDetailsMapper->mapToDomainObject($dtString);
        } catch (\Exception $e) {
            throw new HttpException($e->getCode(), 'You have provided an invalid date or time.');
        }

        if ($timeDetails->getTargetTime() < $timeDetails->getYesterday() || $timeDetails->getNow() < $timeDetails->getTargetTime()) {
            throw new HttpException(422, 'You can only stream the last 24 hours.');
        }

        $audioFileLocation = $this->audioSegmentManager->generateFileLocation($timeDetails);
        if (!file_exists($audioFileLocation)) {
            throw new HttpException(422, 'The selected audio file was not found!');
        }

        try {
            $audioChunk = $this->audioSegmentManager->extractAudioChunk($timeDetails, $audioFileLocation);
        } catch (\Exception $e) {
            throw new HttpException(422, $e->getMessage());
        }

        $response = new Response($audioChunk->getAudioContent());
        $response->setStatusCode(206); // Partial Content
        $response->headers->set('Content-Type', 'audio/mpeg');
        $response->headers->set('Content-Length', (string) ($audioChunk->getEndByte() - $audioChunk->getStartByte() + 1));
        $response->headers->set('Content-Range', \sprintf('bytes %d-%d/%d', $audioChunk->getStartByte(), $audioChunk->getEndByte(), $audioChunk->getTotalFileSize()));
        $response->headers->set('X-Audio-Duration', '3600');
        $response->headers->set('X-Track-Name', $audioChunk->getAudioFileName());

        return $response;
    }

    #[
        Route(
            path: '/schedule',
            name: 'schedule',
            methods: ['GET'],
            priority: 10,
        ),
    ]
    public function schedule(): JsonResponse
    {
        $audioFiles = $this->yammatStreamManager->fetchFilesList();
        $schedule = $this->yammatStreamManager->fetchSchedule();

        $result = [
            'audioFiles' => $audioFiles,
            'schedule' => $schedule,
        ];

        return new JsonResponse($result, Response::HTTP_OK);
    }
}
