<?php

declare(strict_types=1);

namespace Core\Framework\Command;

use Core\Framework\Command\Exception\CommandHandlerNotFoundException;
use Core\Framework\Command\Handler\CommandHandler;
use Core\General\Exception\RuntimeException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\DependencyInjection\Exception\ServiceCircularReferenceException;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;
use Symfony\Component\HttpKernel\Exception\ServiceUnavailableHttpException;

class CommandBus
{
    /** @var array<string, string> */
    private array $commandHandlerMap = [];

    public function __construct(
        private readonly ContainerInterface $locator)
    {
    }

    /**
     * @throws CommandHandlerNotFoundException
     * @throws RuntimeException
     */
    public function handle(Command $command, bool $processedInBackground = true): void
    {
        $className = $command::class;

        if (!isset($this->commandHandlerMap[$className])) {
            throw new CommandHandlerNotFoundException(\sprintf('Command handler for %s not found', $className));
        }

        if ($processedInBackground && $command instanceof BackgroundCommand) {
            $this->handleInBackground($command);

            return;
        }

        try {
            /** @var CommandHandler|null $commandHandler */
            $commandHandler = $this->locator->get($this->commandHandlerMap[$className]);

            $commandHandler?->handle($command);
        } catch (ServiceCircularReferenceException|ServiceNotFoundException|ServiceUnavailableHttpException $e) {
            throw RuntimeException::becauseUnexpectedErrorOccurred($e->getMessage(), $e->getCode());
        }
    }

    /**
     * @throws \Exception
     */
    public function registerCommandMap(CommandMap $commandMap): void
    {
        /**
         * One command handler corresponds to one command, if there will be a situation
         * that more than one handler will handle the same command, we will handle through
         * aggregated handlers, i.e. inside one handler, trigger two different commands
         * (command map will not be used then and specific event will result in different handling)
         */
        $commandMap = $commandMap->getCommandMap();
        foreach ($commandMap as $command => $handler) {
            if (isset($this->commandHandlerMap[$command])) {
                throw new \Exception(\sprintf('Handler for command `%s` already registered', $command));
            }

            $this->commandHandlerMap[$command] = $handler;
        }
    }

    private function handleInBackground(mixed $command): void
    {
        // todo implement background jobs when needed (queue)
    }
}
